<?php

namespace App\Http\Controllers;

use App\Models\Player;
use App\Models\Prize;
use App\Models\PrizeDistribution;
use App\Models\PrizeDistributionConfiguration;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\View;

class WebAdminController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    public function login(Request $request)
    {
        // validate the form entry
        $validator = Validator::make(
            request()->all(),
            [
                'userName' => ['required', 'string', 'max:255'],
                'password' => ['required', 'string', 'max:255',]
            ],
            [
                'userName.required' => 'Please enter your username',
                'email.required' => 'Please enter your password',
            ]
        );
        if ($validator->passes()) {
            // check if the login details match
            if (
                Str::lower($request->userName) == config('auth.username') &&
                $request->password == config('auth.password')
            ) {
                // need to set the session
                session([
                    'admin.username' => $request->userName,
                    'admin.password' => $request->password
                ]);

                return redirect()->route('admin.dashboard');
            }
        }

        $validator->errors()->add('userName', 'Invalid credentials!');
        return redirect()->route('admin.login')->withErrors($validator)->withInput();
    }

    public function index(Request $request)
    {
        // get the list of total prizes distributed
        $prizes = Prize::all();
        $totalCounts = [
            'Available' => 0,
            'Allocated' => 0
        ];

        $totalPrizes = $prizes->map(function ($prize) use (&$totalCounts) {
            $item = $prize->toArray();
            $item['distribution'] = [];

            // find the prize distribution of this for the day
            $day = 1;
            $allocated_this_day = PrizeDistribution::findTotalPrizeDistributedPerDay(
                $prize->id, $day
            );

            array_push($item['distribution'], array(
                'day' => $day,
                'allocated' => $allocated_this_day
            ));

            if ($prize->id != 9) {
                $totalCounts['Available'] += $prize->max_allowed_total;
                $totalCounts['Allocated'] += $prize->total_allocated_count;
            }
            return $item;
        })->toArray();

        //dd($totalPrizes);
        // get the configuration
        $configurations = PrizeDistributionConfiguration::all()->toArray();

        // find the last prize distributed
        $lastPrizeDistributedId = PrizeDistributionConfiguration::findKeyValue('LastPrizeDistributedId');
        if ($lastPrizeDistributedId != 0)
            $lastPrize = Prize::where('id', $lastPrizeDistributedId)->first()->name;
        else
            $lastPrize = '-';

        // get total players
        $totalPlayers = Player::count();

        if ($request->ajax()) {
            return View::make("web-admin._stats", compact('totalPrizes', 'configurations', 'lastPrize', 'totalCounts'));
        }

        return view("web-admin.dashboard", compact('totalPrizes', 'configurations', 'lastPrize', 'totalCounts'));
    }

    public function prizeStimulator($day)
    {
        $logs = [];
        $prizeWon = Prize::findRandomPrizeStimulator($day, $logs);

        // store this in the system
        DB::transaction(function () use ($prizeWon, $day) {
            //store for distribution
            $now = Carbon::now();
            PrizeDistribution::create([
                'prize_id' => $prizeWon['id'],
                'selected_day' => $day,
                'created_at' => $now
            ]);

            // update the prizes table with the count
            Prize::find($prizeWon['id'])->increment('total_allocated_count');

            // store the last prize distributed
            PrizeDistributionConfiguration::setKeyValue('LastPrizeDistributedId', $prizeWon['id']);
        });

        return response()->json($prizeWon);
    }

    public function resetPrizes()
    {
        // delete the prize distribution
        PrizeDistribution::truncate();

        // reset allocated count
        Prize::query()->update(['total_allocated_count' => 0]);

        // delete the players also
        Player::query()->delete();

        return redirect()->route('admin.dashboard');
    }

    public function getReport()
    {
        $players = Player::leftJoin('prizes', function ($join) {
            $join->on('players.prize_id', '=', 'prizes.id');
        })->leftJoin('prize_distribution', function ($join) {
            $join->on('players.prize_id', '=', 'prize_distribution.prize_id')
                ->on('players.id', '=', 'prize_distribution.player_id');
        })
            ->select('players.*', 'prizes.name as prize_name', 'prize_distribution.created_at as prize_distribution_created_at')
            ->get();

        $fileName = 'players_' . date('d/m/Y');

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=' . $fileName . '.csv');

        $output = fopen('php://output', 'w');
        $names =   array('Name', 'Prize', 'Prize Distributed At');
        fputcsv($output, $names);

        if (isset($players) && !empty($players)) {
            foreach ($players as $data) {
                fputcsv($output, array(
                    ucfirst($data['name']),
                    $data['prize_name'],
                    $data['prize_distribution_created_at']
                ));
            }
        }
    }
}
